define(['angular', 'app'], function (angular, app) {
    'use strict';

    app.service('FitnessActivityGraphService', function ($filter, TrackersConstants) {


        var service = {};

        var getChartConfig = function(section, title) {
            var chartConfig = {
                options: {
                    chart: {
                        type: 'spline',
                        borderColor: '#EBBA95',
                        borderWidth: 2,
                        borderRadius: 3,
                        zoomType: 'x'
                    },
                    plotOptions: {
                        series: {
                            marker: { radius: 6 },
                            events: {
                                legendItemClick: function () {
                                    return false;
                                }
                            }
                        }
                    }
                },
                useHighStocks: false,
                title: { text: title },
                credits: { enabled: false },
                loading: false,
                legend: { enabled: true }
            };

            return chartConfig;
        };

        var getDateUTC = function(date, truncateTime) {
            return Date.UTC(
                parseInt($filter('date')(date, 'yyyy')),
                parseInt($filter('date')(date, 'MM')) - 1,
                parseInt($filter('date')(date, 'dd')),
                truncateTime ? 0 : parseInt($filter('date')(date, 'HH')),
                truncateTime ? 0 : parseInt($filter('date')(date, 'mm'))
            );
        };

        var initTmpSeries = function(array) {
            var tmp = {};
            angular.forEach(array, function(item) {
                tmp[item] = [];
            });
            return tmp;
        };

        function getSeriesValue(valueQuantity, emptyValue) {



            return (valueQuantity && angular.isDefined(valueQuantity.value) && (angular.isNumber(valueQuantity.value) || valueQuantity.value.length > 0))
                ? parseFloat(valueQuantity.value) : emptyValue;
        }



        var processFitnessActivitySeries =  function(points,metaData) {

            var tmpSeries = {};

            angular.forEach(angular.copy(points).reverse(), function(point) {
                var dateUTC = getDateUTC(new Date(point['effectiveDateTime'])),
                    component = point.component;

                angular.forEach(component, function(series, seriesIndex){
                    if (angular.isDefined(metaData[seriesIndex])) {
                        var seriesName;
                        var seriesValue;
                        if (seriesIndex === 'steps') {
                            seriesName = metaData[seriesIndex].label;
                            seriesValue = getSeriesValue(series.valueQuantity, null);
                        }

                        else
                        {
                            seriesName = metaData[seriesIndex].label;
                            seriesValue = getSeriesValue(series.valueString, null);
                        }

                        if (!tmpSeries[seriesName])
                            tmpSeries[seriesName] = [];
                        tmpSeries[seriesName].push([dateUTC, seriesValue]);
                    }
                });
            });

            var chartSeries = [];
            for(var dataType in tmpSeries) {
                chartSeries.push({
                    name: dataType,
                    data: tmpSeries[dataType],
                    showInLegend: true,
                    type: dataType === 'spline'
                });
            }
            return chartSeries;
        };

        var processSingleSeries = function(section, points, metaData) {
            var chartSeries = [{
                name: metaData[0].label,
                type: 'spline',
                data: []
            }];

            angular.forEach(angular.copy(points).reverse(), function(point) {
                var dateUTC = getDateUTC(new Date(point['recorded'] || point['effectiveDateTime'])),
                    seriesValue = getSeriesValue(point.valueQuantity, null);

                if (angular.isNumber(seriesValue)) {
                    chartSeries[0].data.push([dateUTC, seriesValue]);
                }
            });

            return chartSeries;
        };

        var getSeriesConfig = function(section, points, metaData) {
            if (!points || points.length === 0) { return []; }

            if(section === 'fitness-activity') {
                return processFitnessActivitySeries(points, metaData);

            } else {
                return processSingleSeries(section, points, metaData);
            }

        };

        var getXaxisConfig = function (section, filter) {

            var startDate = new Date(filter.startDate),
                endDate = new Date(filter.endDate),
                startDateUTC = Date.UTC(
                    parseInt($filter('date')(startDate, 'yyyy')),
                    parseInt($filter('date')(startDate, 'MM')) - 1,
                    parseInt($filter('date')(startDate, 'dd')),
                    0, 0
                ),
                endDateUTC = Date.UTC(
                    parseInt($filter('date')(endDate, 'yyyy')),
                    parseInt($filter('date')(endDate, 'MM')) - 1,
                    parseInt($filter('date')(endDate, 'dd')) + 1,
                    0, 0
                ),
                title = 'Date/Time';

            return {
                type: 'datetime',
                dateTimeLabelFormats: {
                    millisecond: '%b %e %I:%M %p',
                    second: '%b %e %I:%M %p',
                    minute: '%b %e %I:%M %p',
                    hour: '%b %e %I:%M %p',
                    day: '%b %e',
                    week: '%b %e',
                    month: '%b %Y',
                    year: '%Y'
                },
                minRange: 24 * 3600000,
                minTickInterval: 3600000,
                min: startDateUTC,
                max: endDateUTC,
                title: {
                    text: title
                }
            };
        };

        var getYaxisConfig = function(section, metaData) {
            var style = {'font-weight': 'bold', 'color': 'black', 'font-size': '12px'};

            if(section === 'bp-pulse') {
                return [
                    {min: 0, tickInterval: 25, title: {text: 'Blood Pressure' + ' (' + metaData[0].units + ')', style: style}},
                    {min: 0, max: 25, tickInterval: 25, title: {text: 'Pulse' + ' (' + metaData[2].units + ')', style: style}, opposite: true}
                ];
            }
            return {
                min: 0, max: section === 'mood' ? 10.5 : null,
                endOnTick: section !== 'mood',
                allowDecimals: false,
                tickInterval: section === 'mood' ? 1 : 25,
                title: {text: metaData[0].label + ' (' + metaData[0].units + ')', style: style},
                plotBands: section === 'mood' ? getMoodPlotBands() : null
            };
        };

        var configureTooltip = function (section, metaData) {
            var tooltip = {
                shared: section !== "glucose",
                crosshairs: [true]
            };

            if (section === "fitness-activity") {
                tooltip.formatter = function () {
                    return Highcharts.dateFormat('%A %m/%d/%Y %I:%M %p', this.x) + '<br>' + metaData[0].label + ': <b>' + this.y + '</b> ' + metaData[0].units;
                };
            }
            return tooltip;
        };

        service.getGraphConfig = function(section, filter, points, metaData, sectionTitle) {

            var title = sectionTitle + " Values from " + filter.startDate + " to " + filter.endDate,
                chartConfig = getChartConfig(section, title);

            chartConfig.options.tooltip = configureTooltip(section, metaData);
            chartConfig.series = getSeriesConfig(section, points, metaData);
            chartConfig.xAxis = getXaxisConfig(section, filter);
            chartConfig.yAxis = getYaxisConfig(section, metaData);

            return chartConfig;
        };

        return service;
    });
});